// dashboard.js — Full replacement (Part 1 of 3)
// Built to work with your dashboard.html (see file reference). :contentReference[oaicite:1]{index=1}
// Features:
// - Templates editor
// - Profiles table + queue
// - CSV import (Apply Template OR Upload Messages)
// - Duplicate handling (Option B: Skip OR Overwrite & Restart)
// - ICP scoring, GTM export
// - Notes modal, sorting, pagination

// ======= Utilities & Storage helpers =======
const PAGE_SIZE = 25;
const PROFILES_PAGE_SIZE = 25;

function getLS(key) { return new Promise(r => chrome.storage.local.get(key, o => r(o[key]))); }
function setLS(obj) { return new Promise(r => chrome.storage.local.set(obj, () => r())); }

function nowISO() { return new Date().toISOString(); }
function localDateOnly(dateISO) {
  if (!dateISO) return null;
  const d = new Date(dateISO);
  return d.toLocaleDateString();
}
function todayDateString() { return new Date().toLocaleDateString(); }
// NEW — format scheduled timestamp for dashboard
function formatScheduledDate(ts) {
  if (!ts) return "";
  const d = new Date(ts);
  return d.toLocaleString(undefined, {
    day: "2-digit",
    month: "short",
    hour: "2-digit",
    minute: "2-digit",
    hour12: true
  });
}

function isDateBefore(aISO, bISO) {
  if (!aISO) return false;
  const a = new Date(localDateOnly(aISO)).setHours(0,0,0,0);
  const b = new Date((bISO instanceof Date ? bISO : new Date())).setHours(0,0,0,0);
  return a < b;
}
function isDateEqual(aISO, bISO) {
  if (!aISO) return false;
  const a = localDateOnly(aISO);
  const b = (bISO instanceof Date) ? bISO.toLocaleDateString() : (bISO || todayDateString());
  return a === b;
}

// simple toast (reused from popup)
function showToast(msg) {
  // ephemeral, fallback to alert if DOM not available
  try {
    const el = document.createElement('div');
    el.innerText = msg;
    el.style.position = 'fixed';
    el.style.bottom = '18px';
    el.style.left = '50%';
    el.style.transform = 'translateX(-50%)';
    el.style.background = '#0ea75a';
    el.style.color = '#fff';
    el.style.padding = '8px 12px';
    el.style.borderRadius = '8px';
    el.style.boxShadow = '0 2px 8px rgba(0,0,0,0.2)';
    el.style.zIndex = 9999;
    document.body.appendChild(el);
    setTimeout(() => { el.style.opacity = '0'; setTimeout(()=>el.remove(),300); }, 1600);
  } catch (e) {
    try { console.log('toast:', msg); } catch (e) {}
  }
}

// ======= ICP Scoring =======
function scoreICP(profile, icp) {
  if (!icp || !profile) return 0;
  let score = 0;
  try {
    // industries match
    if (icp.industries && icp.industries.length) {
      const pi = (profile.industry || profile.headline || '').toLowerCase();
      const match = icp.industries.some(i => i && pi.includes(i.toLowerCase()));
      if (match) score += 25;
    }
    // personas
    if (icp.personas && icp.personas.length) {
      const title = (profile.position || profile.headline || '').toLowerCase();
      const match = icp.personas.some(p => p && title.includes(p.toLowerCase()));
      if (match) score += 30;
    }
    // regions
    if (icp.regions && icp.regions.length) {
      const loc = (profile.location || '').toLowerCase();
      const match = icp.regions.some(r => r && loc.includes(r.toLowerCase()));
      if (match) score += 25;
    }
    // company size/stage (optional small boost)
    if (icp.companySize && profile.companySize) {
      if (icp.companySize.includes(profile.companySize)) score += 10;
    }
  } catch (e) {
    // graceful
  }
  return Math.min(score, 100);
}

// Expose for debugging
window.scoreICP = scoreICP;

// ======= Merge tag resolver =======
async function resolveMerge(text, profile) {
  if (!text) return '';
  const settings = (await getLS('settings')) || {};
  const yourName = settings?.yourName || '';
  return String(text)
    .replace(/{{\s*first_name\s*}}/ig, profile.firstName || '')
    .replace(/{{\s*last_name\s*}}/ig, profile.lastName || '')
    .replace(/{{\s*full_name\s*}}/ig, profile.fullName || '')
    .replace(/{{\s*headline\s*}}/ig, profile.headline || '')
    .replace(/{{\s*company\s*}}/ig, profile.company || '')
    .replace(/{{\s*position\s*}}/ig, profile.position || '')
    .replace(/{{\s*location\s*}}/ig, profile.location || '')
    .replace(/{{\s*your_name\s*}}/ig, yourName)
    .replace(/{{\s*today\s*}}/ig, new Date().toLocaleDateString());
}

// ======= CSV Parsing & Normalization helpers =======

// robust CSV parser (handles quotes)
function parseCSV(text) {
  const rows = [];
  let cur = '';
  let row = [];
  let inQuotes = false;
  for (let i = 0; i < text.length; i++) {
    const ch = text[i];
    const nxt = text[i+1];
    if (ch === '"' && inQuotes && nxt === '"') {
      cur += '"';
      i++;
      continue;
    }
    if (ch === '"') {
      inQuotes = !inQuotes;
      continue;
    }
    if (ch === ',' && !inQuotes) {
      row.push(cur);
      cur = '';
      continue;
    }
    if ((ch === '\n' || ch === '\r') && !inQuotes) {
      if (cur !== '' || row.length > 0) {
        row.push(cur);
        rows.push(row);
        row = [];
        cur = '';
      }
      continue;
    }
    cur += ch;
  }
  if (cur !== '' || row.length > 0) {
    row.push(cur);
    rows.push(row);
  }
  return rows.map(r => r.map(c => (c || '').trim()));
}

// normalize header token to match many variants
function normalizeHeaderKey(h) {
  if (!h) return '';
  return String(h).trim().toLowerCase().replace(/\ufeff/g,'').replace(/[^a-z0-9]/gi, '');
}

// map normalized header tokens to canonical keys
function mapHeaderToKey(token) {
  if (!token) return '';
  if (['profileurl','url','linkedinurl','linkedinprofile','liurl','profile'].includes(token)) return 'profileUrl';
  if (['firstname','first','givenname'].includes(token)) return 'firstName';
  if (['lastname','last','surname'].includes(token)) return 'lastName';
  if (['fullname','name','fullName'.toLowerCase()].includes(token)) return 'fullName';
  if (['company','org','organization'].includes(token)) return 'company';
  if (['position','title','role'].includes(token)) return 'position';
  if (['location','city','region'].includes(token)) return 'location';
  if (['email','emailaddress'].includes(token)) return 'email';
  if (['notes','note'].includes(token)) return 'notes';
  if (['templateused','template','tpl'].includes(token)) return 'templateUsed';
  if (['msg1','message1','messageone','msgone','m1'].includes(token)) return 'msg1';
  if (['msg2','message2','messagetwo','msgtwo','m2'].includes(token)) return 'msg2';
  if (['msg3','message3','messagethree','msgthree','m3'].includes(token)) return 'msg3';
  return token; // fallback: keep raw normalized token
}

// build messages from a parsed row object (either apply template or use CSV messages)
function buildMessagesFromRow(rowObj, messageMode, chosenTemplate, templates) {
  if (messageMode === 'template') {
    const tplName = chosenTemplate || (rowObj.templateUsed || '');
    if (tplName && templates && templates[tplName]) {
      return JSON.parse(JSON.stringify(templates[tplName].messages || {}));
    }
    return {};
  } else {
    return {
      msg1: (rowObj.msg1 || '').toString(),
      msg2: (rowObj.msg2 || '').toString(),
      msg3: (rowObj.msg3 || '').toString()
    };
  }
}

// ======= DOM & State =======
document.addEventListener('DOMContentLoaded', async () => {
  // ensure base keys
  if (!await getLS('templates')) await setLS({ templates: {} });
  if (!await getLS('profiles')) await setLS({ profiles: {} });
  if (!await getLS('settings')) await setLS({ settings: {} });
 
  // dom refs (matches dashboard.html)
  const statsTotalEl = document.getElementById('statsTotal');
  const statsL1El = document.getElementById('statsL1');
  const statsL2El = document.getElementById('statsL2');
  const statsL3El = document.getElementById('statsL3');
  const tplDelayDays = document.getElementById('tplDelayDays');

  const queueBody = document.getElementById('queueBody');
  const prevPageBtn = document.getElementById('prevPage');
  const nextPageBtn = document.getElementById('nextPage');
  const pageInfoEl = document.getElementById('pageInfo');

  const tplNameEl = document.getElementById('tplName');
  const tplMsg1El = document.getElementById('tplMsg1');
  const tplMsg2El = document.getElementById('tplMsg2');
  const tplMsg3El = document.getElementById('tplMsg3');
  const saveTemplateBtn = document.getElementById('saveTemplate');
  const tplListEl = document.getElementById('tplList');

  const profilesBody = document.getElementById('profilesBody');
  const exportCsvBtn = document.getElementById('exportCsv');
  const importFileEl = document.getElementById('importFile');
  const downloadGtmBtn = document.getElementById('downloadGtmBtn');

  const profilesSearchEl = document.getElementById('profilesSearch');
  const profilesPrevBtn = document.getElementById('profilesPrev');
  const profilesNextBtn = document.getElementById('profilesNext');
  const profilesPageInfoEl = document.getElementById('profilesPageInfo');

  // CSV refs
  const csvUploadEl = document.getElementById('csvUpload');
  const importCsvBtn = document.getElementById('importCsvBtn');
  const downloadTemplateBtn = document.getElementById('downloadTemplateBtn');
  const csvModeSelect = document.getElementById('csvModeSelect');
  const csvTemplateSelect = document.getElementById('csvTemplateSelect');

  // notes modal
  const notesModalBackdrop = document.getElementById('notesModalBackdrop');
  const notesModalClose = document.getElementById('notesModalClose');
  const notesModalCancel = document.getElementById('notesModalCancel');
  const notesModalSave = document.getElementById('notesModalSave');
  const notesModalTextarea = document.getElementById('notesModalTextarea');
  const notesModalProfileUrl = document.getElementById('notesModalProfileUrl');

  // duplicates modal
  const csvDuplicateModal = document.getElementById('csvDuplicateModal');
  const csvDuplicateClose = document.getElementById('csvDuplicateClose');
  const csvDuplicateSummary = document.getElementById('csvDuplicateSummary');
  const csvDuplicateList = document.getElementById('csvDuplicateList');
  const csvSkipDuplicates = document.getElementById('csvSkipDuplicates');
  const csvOverwriteDuplicates = document.getElementById('csvOverwriteDuplicates');
  const csvRestartDuplicates = document.getElementById('csvRestartDuplicates');

  // hide the middle overwrite button per Option B
  if (csvOverwriteDuplicates) csvOverwriteDuplicates.style.display = 'none';

  // state
  let currentPage = 1;
  let currentQueue = [];

  let profilesPage = 1;
  let profilesFiltered = [];
  let profilesSort = { column: null, direction: null };

  let icpSettings = null;

  let __pendingImportRows = null;
  let __pendingImportDuplicates = null;
  let __pendingMessageMode = 'template';
  let __pendingChosenTemplate = '';
  // ======= Main Refresh =======
  async function refreshAll() {
    if (profilesSearchEl) profilesSearchEl.value = "";
	icpSettings = await getLS("icpSettings");
    await renderStats();
    await buildQueue();
    await renderQueue();
    await renderTemplates();
    await renderProfilesTable();
    await updateGtmButton();
    await renderIcpSummary();
	if (Object.keys(await getLS("profiles") || {}).length === 0) {
  showToast("No profiles yet");
}

  }

  // ======= ICP Summary Section =======
  async function renderIcpSummary() {
    const icp = (await getLS('icpSettings')) || null;
    const industriesEl = document.getElementById('icpIndustries');
    const regionsEl = document.getElementById('icpRegions');
    const personasEl = document.getElementById('icpPersonas');
    const goalEl = document.getElementById('icpGoal');
    const offerEl = document.getElementById('icpOffer');

    if (!icp) {
      if (industriesEl) industriesEl.innerText = 'Industries: —';
      if (regionsEl) regionsEl.innerText = 'Regions: —';
      if (personasEl) personasEl.innerText = 'Personas: —';
      if (goalEl) goalEl.innerText = 'Goal: —';
      if (offerEl) offerEl.innerText = '—';
      return;
    }

    const industries = icp.industries?.length ? icp.industries.join(', ') : '—';
    const regions = icp.regions?.length ? icp.regions.join(', ') : (icp.customRegion || '—');
    const personas = icp.personas?.length ? icp.personas.join(', ') : (icp.customPersona || '—');
    const goal = icp.goal || '—';
    const offerSummary = icp.offer?.service ? `${icp.offer.service} • ${icp.offer.cta || ''}` : '—';

    if (industriesEl) industriesEl.innerText = `Industries: ${industries}`;
    if (regionsEl) regionsEl.innerText = `Regions: ${regions}`;
    if (personasEl) personasEl.innerText = `Personas: ${personas}`;
    if (goalEl) goalEl.innerText = `Goal: ${goal}`;
    if (offerEl) offerEl.innerText = offerSummary;
  }

  document.getElementById('editIcpBtn')?.addEventListener('click', () => {
    chrome.tabs.create({ url: chrome.runtime.getURL('icp_setup.html') });
  });

  // ======= STATS =======
  async function renderStats() {
    if (!statsTotalEl) return;
	const profiles = (await getLS('profiles')) || {};
    const today = todayDateString();
    let l1 = 0, l2 = 0, l3 = 0;

    Object.values(profiles).forEach(p => {
      if (p.sentDates?.msg1 && isDateEqual(p.sentDates.msg1, today)) l1++;
      if (p.sentDates?.msg2 && isDateEqual(p.sentDates.msg2, today)) l2++;
      if (p.sentDates?.msg3 && isDateEqual(p.sentDates.msg3, today)) l3++;
    });

    if (statsL1El) statsL1El.innerText = String(l1);
    if (statsL2El) statsL2El.innerText = String(l2);
    if (statsL3El) statsL3El.innerText = String(l3);
    if (statsTotalEl) statsTotalEl.innerText = String(l1 + l2 + l3);
  }

  // ======= QUEUE BUILD =======
  async function buildQueue() {
    const profiles = (await getLS('profiles')) || {};
    const arr = Object.values(profiles).map(p => ({ ...p }));

    // attach ICP score
arr.forEach(p => {
  if (typeof p.icpScore !== "number") {
    try {
      p.icpScore = scoreICP(p, icpSettings);
    } catch {
      p.icpScore = 0;
    }
  }
});

    const today = todayDateString();

const now = new Date();

const queue = arr.filter(p => {
  if (!p.nextDueDate) return false;

  // 🚫 exclude completed sequences
  if (p.sentDates?.msg3) return false;

  return new Date(p.nextDueDate) <= now;
});

    queue.forEach(p => {
      if (!p.sentDates) p.sentDates = { msg1: null, msg2: null, msg3: null };
      if (!p.sentDates.msg1) p.nextMessage = "Message 1";
      else if (!p.sentDates.msg2) p.nextMessage = "Message 2";
      else if (!p.sentDates.msg3) p.nextMessage = "Message 3";
      else p.nextMessage = "Completed";

      p.dueDateDisplay = localDateOnly(p.nextDueDate) || '';
      p.isOverdue = isDateBefore(p.nextDueDate, today);
    });

    // sort: overdue first → due date → icpscore → name
    queue.sort((a, b) => {
      if (a.isOverdue && !b.isOverdue) return -1;
      if (!a.isOverdue && b.isOverdue) return 1;

      const ad = new Date(localDateOnly(a.nextDueDate)).getTime();
      const bd = new Date(localDateOnly(b.nextDueDate)).getTime();
      if (ad !== bd) return ad - bd;

      if ((b.icpScore || 0) !== (a.icpScore || 0)) return (b.icpScore || 0) - (a.icpScore || 0);

      const an = (a.fullName || a.profileUrl || '').toLowerCase();
      const bn = (b.fullName || b.profileUrl || '').toLowerCase();
      return an.localeCompare(bn);
    });

    currentQueue = queue;
    currentPage = 1;
  }

  // ======= QUEUE RENDER =======
  function renderQueue() {
	if (!queueBody) return;
    const total = currentQueue.length;
    const pages = Math.max(1, Math.ceil(total / PAGE_SIZE));
    if (currentPage > pages) currentPage = pages;

    const start = (currentPage - 1) * PAGE_SIZE;
    const items = currentQueue.slice(start, start + PAGE_SIZE);

    queueBody.innerHTML = "";

    items.forEach(p => {
      const tr = document.createElement("tr");

      // ICP color row
      if (p.icpScore >= 70) tr.classList.add("icp-high-row");
      else if (p.icpScore >= 40) tr.classList.add("icp-mid-row");
      else tr.classList.add("icp-low-row");

      if (p.isOverdue) tr.classList.add("overdue");

      const badge =
        p.icpScore >= 70
          ? `<span class="icpBadge high">${p.icpScore}</span>`
          : p.icpScore >= 40
          ? `<span class="icpBadge mid">${p.icpScore}</span>`
          : `<span class="icpBadge low">${p.icpScore}</span>`;

      tr.innerHTML = `
        <td>
          ${p.fullName || ''}
          ${badge}
          <div class="small">${p.profileUrl || ''}</div>
        </td>
        <td>${p.templateUsed || ''}</td>
        <td>${p.nextMessage}</td>
        <td>${p.dueDateDisplay}</td>
        <td>${p.isOverdue ? '<strong style="color:#c00">❗ Overdue</strong>' : 'Due Today'}</td>
      `;

      const tdActions = document.createElement("td");

      // Open & Copy
      const openBtn = document.createElement("button");
      openBtn.className = "btn";
      openBtn.innerText = "Open & Copy Next Message";
      openBtn.onclick = () => openAndCopyMessage(p);

      const markBtn = document.createElement("button");
      markBtn.className = "btn";
      markBtn.innerText = "Mark Sent";
      markBtn.onclick = () => markAsSent(p);

      const delBtn = document.createElement("button");
      delBtn.className = "btn";
      delBtn.innerText = "Delete";
      delBtn.onclick = () => deleteProfile(p.profileUrl);

      tdActions.appendChild(openBtn);
      tdActions.appendChild(markBtn);
      tdActions.appendChild(delBtn);
      tr.appendChild(tdActions);

      queueBody.appendChild(tr);
    });

    pageInfoEl.innerText = `Page ${currentPage} of ${pages}`;
    prevPageBtn.disabled = currentPage <= 1;
    nextPageBtn.disabled = currentPage >= pages;
  }

  prevPageBtn.onclick = () => { if (currentPage > 1) { currentPage--; renderQueue(); }};
  nextPageBtn.onclick = () => {
    const pages = Math.max(1, Math.ceil(currentQueue.length / PAGE_SIZE));
    if (currentPage < pages) { currentPage++; renderQueue(); }
  };

  async function openAndCopyMessage(p) {
    const profiles = (await getLS("profiles")) || {};
    const rec = profiles[p.profileUrl];
    if (!rec) return alert("Record missing.");

    const msgKey = !rec.sentDates.msg1
      ? "msg1"
      : !rec.sentDates.msg2
      ? "msg2"
      : !rec.sentDates.msg3
      ? "msg3"
      : null;

    if (!msgKey) return alert("Sequence complete.");

    const resolved = await resolveMerge(rec.messages[msgKey], rec);

    chrome.tabs.create({ url: p.profileUrl }, async (tab) => {
      await new Promise(r => setTimeout(r, 600));

      function waitContent(tabId, retries = 25) {
        return new Promise(async resolve => {
          for (let i = 0; i < retries; i++) {
            try {
              const res = await chrome.tabs.sendMessage(tabId, { type: "ping" });
              if (res && res.ok) return resolve(true);
            } catch {}
            await new Promise(r => setTimeout(r, 300));
          }
          resolve(false);
        });
      }

      const ready = await waitContent(tab.id);
      if (!ready) return alert("Content script not ready.");

      chrome.tabs.sendMessage(
        tab.id,
        { action: "copyText", text: resolved },
        () => alert("Message copied.")
      );
    });
  }

async function markAsSent(p) {
  const profiles = (await getLS("profiles")) || {};
  const rec = profiles[p.profileUrl];
  if (!rec) return;

  const templates = (await getLS("templates")) || {};
  const tpl = templates[rec.templateUsed] || {};
  const delayDays = Number(tpl.delayDays ?? rec.sequenceDelayDays ?? 3);

  const now = new Date();
  const nowISO = now.toISOString();
  const nextDue = new Date(
    now.getTime() + delayDays * 86400000
  ).toISOString();

  if (!rec.sentDates.msg1) {
    rec.sentDates.msg1 = nowISO;
    rec.status = "msg1_sent";
    rec.nextDueDate = nextDue;
  }
  else if (!rec.sentDates.msg2) {
    rec.sentDates.msg2 = nowISO;
    rec.status = "msg2_sent";
    rec.nextDueDate = nextDue;
  }
  else if (!rec.sentDates.msg3) {
    rec.sentDates.msg3 = nowISO;
    rec.status = "completed";
    rec.nextDueDate = null;
  }
  else {
    return alert("All messages already sent.");
  }

  rec.lastUpdated = nowISO;
  profiles[p.profileUrl] = rec;
  await setLS({ profiles });
  await refreshAll();
}

  async function deleteProfile(url) {
    const c = confirm("Delete saved sequence?");
    if (!c) return;
    const profiles = (await getLS("profiles")) || {};
    delete profiles[url];
    await setLS({ profiles });
    await refreshAll();
  }

  // ======= TEMPLATES LIST =======
  async function renderTemplates() {
    if (!tplListEl) return;
	const templates = (await getLS("templates")) || {};

    tplListEl.innerHTML = "";

    const names = Object.keys(templates).sort();
    if (names.length === 0) {
      tplListEl.innerHTML = "<div class='small'>No templates</div>";
      if (csvTemplateSelect) {
  csvTemplateSelect.style.display = "none";
}
      return;
    }

    names.forEach(name => {
      const t = templates[name];
      const div = document.createElement("div");
      div.style.padding = "8px";
      div.style.borderBottom = "1px solid #eee";

      div.innerHTML = `
        <div style="font-weight:600">${name}</div>
        <div class="small">${(t.messages.msg1 || "").slice(0, 120)}</div>
      `;

      const wrap = document.createElement("div");
      wrap.style.marginTop = "6px";

      const applyBtn = document.createElement("button");
      applyBtn.className = "btn";
      applyBtn.innerText = "Apply";
      applyBtn.onclick = async () => {
        const url = prompt("Paste LinkedIn profile URL:");
        if (!url) return;
        const profiles = (await getLS("profiles")) || {};
        profiles[url] = {
          profileUrl: url,
          dateCaptured: nowISO(),
          lastUpdated: nowISO(),
          fullName: "",
          firstName: "",
          lastName: "",
          headline: "",
          company: "",
          position: "",
          location: "",
          connectionStatus: "unknown",
          templateUsed: name,
          messages: JSON.parse(JSON.stringify(t.messages)),
          sentDates: { msg1: null, msg2: null, msg3: null },
          nextDueDate: nowISO(), // Message 1 due immediately
          status: "new",
          notes: "",
          email: "",
          gtmExportedDate: null
        };
        await setLS({ profiles });
        await refreshAll();
      };

      const editBtn = document.createElement("button");
      editBtn.className = "btn";
      editBtn.innerText = "Edit";
      editBtn.onclick = () => {
  tplNameEl.value = name;
  tplMsg1El.value = t.messages?.msg1 || "";
  tplMsg2El.value = t.messages?.msg2 || "";
  tplMsg3El.value = t.messages?.msg3 || "";

  // ✅ RESTORE DELAY (THIS WAS MISSING)
  if (tplDelayDays) {
    tplDelayDays.value = Number(t.delayDays ?? 3);
  }
};


      const delBtn = document.createElement("button");
      delBtn.className = "btn";
      delBtn.innerText = "Delete";
      delBtn.onclick = async () => {
        if (!confirm(`Delete template ${name}?`)) return;
        delete templates[name];
        await setLS({ templates });
        await renderTemplates();
      };

      wrap.appendChild(applyBtn);
      wrap.appendChild(editBtn);
      wrap.appendChild(delBtn);

      div.appendChild(wrap);
      tplListEl.appendChild(div);
    });

    // populate template dropdown for CSV import
    csvTemplateSelect.innerHTML = '<option value="">— choose template —</option>';
    names.forEach(name => {
      const opt = document.createElement("option");
      opt.value = name;
      opt.innerText = name;
      csvTemplateSelect.appendChild(opt);
    });

    if (csvModeSelect.value === "applyTemplate") {
      csvTemplateSelect.style.display = "inline-block";
    } else {
      csvTemplateSelect.style.display = "none";
    }
  }

 saveTemplateBtn.onclick = async () => {
  const name = (tplNameEl.value || "").trim();
  if (!name) return alert("Template name required");

  const templates = (await getLS("templates")) || {};

  templates[name] = {
    messages: {
      msg1: tplMsg1El.value,
      msg2: tplMsg2El.value,
      msg3: tplMsg3El.value
    },

    // ✅ single delay applied between all messages
    delayDays: Number(tplDelayDays?.value || 3),

    updated: nowISO()
  };

  await setLS({ templates });

  // clear editor
  tplNameEl.value = "";
  tplMsg1El.value = "";
  tplMsg2El.value = "";
  tplMsg3El.value = "";
  if (tplDelayDays) tplDelayDays.value = 3;

  await renderTemplates();
  showToast("Template saved");
};


  // toggle template dropdown
  csvModeSelect.onchange = () => {
    csvTemplateSelect.style.display =
      csvModeSelect.value === "applyTemplate" ? "inline-block" : "none";
  };
  // ======= PROFILES TABLE =======
  async function renderProfilesTable() {
	if (!profilesBody) return;
    const profiles = (await getLS("profiles")) || {};
    const search = (profilesSearchEl.value || "").toLowerCase();

    // attach ICP to all
const arr = Object.values(profiles).map(p => ({
  ...p,
  icpScore:
    typeof p.icpScore === "number"
      ? p.icpScore
      : scoreICP(p, icpSettings)
}));


    profilesFiltered = arr.filter(p => {
      const block = [
        p.fullName, p.firstName, p.lastName,
        p.company, p.position, p.profileUrl,
        p.templateUsed, p.notes, p.email
      ].join(" ").toLowerCase();
      return block.includes(search);
    });

    // sorting
    if (profilesSort.column) {
      profilesFiltered.sort((a, b) => {
        const col = profilesSort.column;
        const dir = profilesSort.direction === "asc" ? 1 : -1;
        let A = "", B = "";

        switch (col) {
          case "captured": A = a.dateCaptured || ""; B = b.dateCaptured || ""; break;
          case "name": A = (a.fullName || a.profileUrl || ""); B = (b.fullName || b.profileUrl || ""); break;
          case "template": A = a.templateUsed || ""; B = b.templateUsed || ""; break;
          case "status": A = a.status || ""; B = b.status || ""; break;
          case "msg1": A = a.sentDates?.msg1 || ""; B = b.sentDates?.msg1 || ""; break;
          case "msg2": A = a.sentDates?.msg2 || ""; B = b.sentDates?.msg2 || ""; break;
          case "msg3": A = a.sentDates?.msg3 || ""; B = b.sentDates?.msg3 || ""; break;
          case "email": A = a.email || ""; B = b.email || ""; break;
          case "icp": A = String(a.icpScore || ""); B = String(b.icpScore || ""); break;
        }

        A = A.toString().toLowerCase();
        B = B.toString().toLowerCase();

        if (A < B) return -1 * dir;
        if (A > B) return 1 * dir;
        return 0;
      });
    } else {
      profilesFiltered.sort((a, b) =>
        (a.fullName || a.profileUrl || "").toLowerCase()
          .localeCompare((b.fullName || b.profileUrl || "").toLowerCase())
      );
    }

    // pagination
    const total = profilesFiltered.length;
    const pages = Math.max(1, Math.ceil(total / PROFILES_PAGE_SIZE));
    if (profilesPage > pages) profilesPage = pages;
    const start = (profilesPage - 1) * PROFILES_PAGE_SIZE;
    const rows = profilesFiltered.slice(start, start + PROFILES_PAGE_SIZE);

    profilesBody.innerHTML = "";

    rows.forEach(p => {
      const tr = document.createElement("tr");

      if (p.icpScore >= 70) tr.classList.add("icp-high-row");
      else if (p.icpScore >= 40) tr.classList.add("icp-mid-row");
      else tr.classList.add("icp-low-row");

      if (isDateBefore(p.nextDueDate, todayDateString())) tr.classList.add("overdue");

      tr.innerHTML = `
  <td>${p.dateCaptured ? p.dateCaptured.split("T")[0] : ""}</td>
  <td><a href="${p.profileUrl}" target="_blank">${p.fullName || p.profileUrl}</a></td>
  <td>${p.templateUsed || ""}</td>
  <td>${p.email || ""}</td>
  <td>${p.status || ""}</td>
  <td style="text-align:center">${p.sentDates?.msg1 ? "✔" : ""}</td>
  <td style="text-align:center">${p.sentDates?.msg2 ? "✔" : ""}</td>
  <td style="text-align:center">${p.sentDates?.msg3 ? "✔" : ""}</td>
  <td style="text-align:center">${p.icpScore || ""}</td>
  <td class="notes-cell" title="${p.notes || ""}">${(p.notes || "").slice(0, 60)}</td>

  <!-- NEW: Scheduled column -->
  <td>${formatScheduledDate(p.nextDueDate)}</td>

  <td>
    <button class="btn editP" data-url="${p.profileUrl}">Edit</button>
    <button class="btn noteP" data-url="${p.profileUrl}">Notes</button>
    <button class="btn delP" data-url="${p.profileUrl}">Delete</button>
  </td>
`;
      profilesBody.appendChild(tr);
    });

    profilesPageInfoEl.innerText = `Page ${profilesPage} of ${pages}`;
    profilesPrevBtn.disabled = profilesPage <= 1;
    profilesNextBtn.disabled = profilesPage >= pages;

    document.querySelectorAll(".delP").forEach(btn => {
      btn.onclick = async e => {
        const url = e.target.dataset.url;
        if (!confirm("Delete profile?")) return;
        const profiles = (await getLS("profiles")) || {};
        delete profiles[url];
        await setLS({ profiles });
        await refreshAll();
      };
    });

    document.querySelectorAll(".editP").forEach(btn => {
      btn.onclick = async e => {
        const url = e.target.dataset.url;
        const profiles = (await getLS("profiles")) || {};
        const p = profiles[url];
        if (!p) return alert("Profile missing");
        const newEmail = prompt("Email:", p.email || "");
        if (newEmail !== null) p.email = newEmail.trim();
        p.lastUpdated = nowISO();
        profiles[url] = p;
        await setLS({ profiles });
        await renderProfilesTable();
      };
    });

    document.querySelectorAll(".noteP").forEach(btn => {
      btn.onclick = async e => {
        const url = e.target.dataset.url;
        const profiles = (await getLS("profiles")) || {};
        const p = profiles[url];
        if (!p) return alert("Profile missing");
        openNotesModal(url, p.notes || "");
      };
    });
  }

  profilesPrevBtn.onclick = () => {
    if (profilesPage > 1) { profilesPage--; renderProfilesTable(); }
  };
  profilesNextBtn.onclick = () => {
    const pages = Math.ceil(profilesFiltered.length / PROFILES_PAGE_SIZE);
    if (profilesPage < pages) { profilesPage++; renderProfilesTable(); }
  };

  profilesSearchEl.oninput = () => {
    profilesPage = 1;
    renderProfilesTable();
  };

  // table header sorting
  document.querySelectorAll(".sortable").forEach(th => {
    th.onclick = () => {
      const col = th.dataset.sort;
      if (!col) return;

      if (profilesSort.column !== col) {
        profilesSort = { column: col, direction: "asc" };
      } else {
        profilesSort.direction =
          profilesSort.direction === "asc" ? "desc" : "asc";
      }

      document.querySelectorAll(".sortable").forEach(h => {
        h.classList.remove("sorted-asc", "sorted-desc");
      });

      th.classList.add(
        profilesSort.direction === "asc" ? "sorted-asc" : "sorted-desc"
      );

      profilesPage = 1;
      renderProfilesTable();
    };
  });

  // ===== Notes Modal =====
  let notesModalCurrentUrl = null;

  function openNotesModal(url, notes) {
    notesModalCurrentUrl = url;
    notesModalProfileUrl.innerText = url;
    notesModalTextarea.value = notes;
    notesModalBackdrop.style.display = "flex";
    notesModalBackdrop.setAttribute("aria-hidden", "false");
    notesModalTextarea.focus();
  }

  function closeNotesModal() {
    notesModalCurrentUrl = null;
    notesModalBackdrop.style.display = "none";
    notesModalBackdrop.setAttribute("aria-hidden", "true");
  }

  notesModalClose.onclick = closeNotesModal;
  notesModalCancel.onclick = closeNotesModal;

  notesModalSave.onclick = async () => {
    if (!notesModalCurrentUrl) return closeNotesModal();
    const profiles = (await getLS("profiles")) || {};
    const p = profiles[notesModalCurrentUrl];
    if (!p) return alert("Profile missing");
    p.notes = notesModalTextarea.value;
    p.lastUpdated = nowISO();
    profiles[notesModalCurrentUrl] = p;
    await setLS({ profiles });
    closeNotesModal();
    await renderProfilesTable();
  };

  notesModalBackdrop.onclick = e => {
    if (e.target === notesModalBackdrop) closeNotesModal();
  };

  // ===== GTM Pending Export =====
  async function getGtmPending() {
    const profiles = (await getLS("profiles")) || {};
    return Object.values(profiles).filter(
      p => p.sentDates?.msg1 && p.sentDates?.msg2 && p.sentDates?.msg3 && !p.gtmExportedDate
    );
  }

  async function updateGtmButton() {
    const pending = await getGtmPending();
    downloadGtmBtn.innerText = `Download GTM Pending (${pending.length})`;
  }

  downloadGtmBtn.onclick = async () => {
    const pending = await getGtmPending();
    if (pending.length === 0) return alert("Nothing pending.");

    const rows = [
      [
        "dateCaptured","profileUrl","fullName","firstName","lastName",
        "headline","company","position","location","email","notes",
        "msg1_date","msg2_date","msg3_date","templateUsed"
      ]
    ];

    pending.forEach(p => {
      rows.push([
        p.dateCaptured || "",
        p.profileUrl || "",
        p.fullName || "",
        p.firstName || "",
        p.lastName || "",
        (p.headline || "").replace(/\n/g, " "),
        p.company || "",
        p.position || "",
        p.location || "",
        p.email || "",
        (p.notes || "").replace(/\n/g, " "),
        p.sentDates?.msg1 || "",
        p.sentDates?.msg2 || "",
        p.sentDates?.msg3 || "",
        p.templateUsed || ""
      ]);
    });

    const csv = rows.map(r =>
      r.map(c => `"${String(c).replace(/"/g, '""')}"`).join(",")
    ).join("\n");

    const blob = new Blob([csv], { type: "text/csv" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "gtm_pending.csv";
    a.click();
    URL.revokeObjectURL(url);

    const profiles = (await getLS("profiles")) || {};
    const now = nowISO();
    pending.forEach(p => profiles[p.profileUrl].gtmExportedDate = now);
    await setLS({ profiles });
    await refreshAll();
  };

  // ===== Export CSV (full export) =====
  exportCsvBtn.onclick = async () => {
    const profiles = (await getLS("profiles")) || {};
    const rows = [[
      "dateCaptured","profileUrl","fullName","firstName","lastName",
      "headline","company","position","location","email","notes",
      "msg1_date","msg2_date","msg3_date","status","templateUsed"
    ]];

    Object.values(profiles).forEach(p => {
      rows.push([
        p.dateCaptured || "",
        p.profileUrl || "",
        p.fullName || "",
        p.firstName || "",
        p.lastName || "",
        (p.headline || "").replace(/\n/g, " "),
        p.company || "",
        p.position || "",
        p.location || "",
        p.email || "",
        (p.notes || "").replace(/\n/g, " "),
        p.sentDates?.msg1 || "",
        p.sentDates?.msg2 || "",
        p.sentDates?.msg3 || "",
        p.status || "",
        p.templateUsed || ""
      ]);
    });

    const csv = rows.map(r =>
      r.map(c => `"${String(c).replace(/"/g, '""')}"`).join(",")
    ).join("\n");

    const blob = new Blob([csv], { type: "text/csv" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "linkedin_sequences_all.csv";
    a.click();
    URL.revokeObjectURL(url);
  };

  // ===== Import JSON =====
  importFileEl.onchange = async e => {
    const file = e.target.files[0];
    if (!file) return;
    try {
      const text = await file.text();
      const obj = JSON.parse(text);
      const profiles = (await getLS("profiles")) || {};
      const templates = (await getLS("templates")) || {};

      if (obj.profiles) Object.assign(profiles, obj.profiles);
      if (obj.templates) Object.assign(templates, obj.templates);

      await setLS({ profiles, templates });
      await refreshAll();
      alert("Import successful.");
    } catch (err) {
      alert("Import failed: " + err.message);
    }
  };

  // ===== CSV IMPORT HANDLER =====
  importCsvBtn.onclick = async () => {
    const file = csvUploadEl.files[0];
    if (!file) return alert("Choose a CSV file.");

    const text = await file.text();
    const parsed = parseCSV(text);
    if (!parsed.length) return alert("CSV empty.");

    const headerRaw = parsed[0].map(h => h || "");
    const normalizedHeader = headerRaw.map(h => normalizeHeaderKey(h));

    // Build object rows
    const rows = [];
    for (let i = 1; i < parsed.length; i++) {
      const row = parsed[i];
      if (!row || row.length === 0) continue;

      const obj = {};

      for (let c = 0; c < headerRaw.length; c++) {
        const rawKey = headerRaw[c];
        const norm = normalizedHeader[c];
        const val = row[c] || "";

        const canonical = mapHeaderToKey(norm);
        obj[canonical] = val;
      }

      if (!obj.profileUrl) continue;
      obj.profileUrl = obj.profileUrl.toString().trim().replace(/\/+$/, "");

      rows.push(obj);
    }

    if (!rows.length) return alert("No valid profileUrl entries found.");

    const messageMode =
      csvModeSelect.value === "uploadMessages" ? "csvMessages" : "template";

    const chosenTemplate =
      csvTemplateSelect.value || "";

    if (messageMode === "template" && !chosenTemplate) {
      if (!confirm("No template selected. Continue?")) return;
    }

    // detect duplicates
    const profiles = (await getLS("profiles")) || {};
    const duplicates = [];
    rows.forEach(r => {
      if (profiles[r.profileUrl]) duplicates.push(r.profileUrl);
    });

    __pendingImportRows = rows;
    __pendingImportDuplicates = [...new Set(duplicates)];
    __pendingMessageMode = messageMode;
    __pendingChosenTemplate = chosenTemplate;

    if (__pendingImportDuplicates.length) {
      csvDuplicateSummary.innerText =
        `Found ${__pendingImportDuplicates.length} duplicate profile(s).`;
      csvDuplicateList.innerHTML = "";
      __pendingImportDuplicates.slice(0, 10).forEach(u => {
        const li = document.createElement("li");
        li.innerText = u;
        csvDuplicateList.appendChild(li);
      });

      csvDuplicateModal.style.display = "flex";
      csvDuplicateModal.setAttribute("aria-hidden", "false");
    } else {
      // no duplicates → import directly with skip mode
      const res = await applyImportRows(
        __pendingImportRows,
        "skip",
        __pendingMessageMode,
        __pendingChosenTemplate
      );
      alert(`Import complete — added ${res.added}, updated ${res.updated}, skipped ${res.skipped}.`);
      __pendingImportRows = null;
      __pendingImportDuplicates = null;
      __pendingMessageMode = "template";
      __pendingChosenTemplate = "";
    }
  };

  // ===== CSV Duplicate Modal =====
  function closeCsvDuplicateModal() {
    __pendingImportRows = null;
    __pendingImportDuplicates = null;
    __pendingMessageMode = "template";
    __pendingChosenTemplate = "";

    csvDuplicateModal.style.display = "none";
    csvDuplicateModal.setAttribute("aria-hidden", "true");
  }

  csvDuplicateClose.onclick = closeCsvDuplicateModal;

  csvSkipDuplicates.onclick = async () => {
    if (!__pendingImportRows) return closeCsvDuplicateModal();
    const res = await applyImportRows(
      __pendingImportRows,
      "skip",
      __pendingMessageMode,
      __pendingChosenTemplate
    );
    alert(`Import complete — added ${res.added}, updated ${res.updated}, skipped ${res.skipped}.`);
    closeCsvDuplicateModal();
  };

  csvRestartDuplicates.onclick = async () => {
    if (!__pendingImportRows) return closeCsvDuplicateModal();
    const res = await applyImportRows(
      __pendingImportRows,
      "restart",
      __pendingMessageMode,
      __pendingChosenTemplate
    );
    alert(`Import complete — added ${res.added}, updated ${res.updated}, skipped ${res.skipped}.`);
    closeCsvDuplicateModal();
  };

// ===== Apply CSV rows (core logic) =====
async function applyImportRows(
  rows,
  mode = "skip",
  messageMode = "template",
  chosenTemplate = ""
) {
  const profiles = (await getLS("profiles")) || {};
  const templates = (await getLS("templates")) || {};

  let added = 0,
    updated = 0,
    skipped = 0;

  for (const r of rows) {
    let url = (r.profileUrl || "").trim();
    if (!url) continue;

    if (!/^https?:\/\//i.test(url)) {
      url = "https://" + url.replace(/^\/+/, "");
    }
    url = url.replace(/\/+$/, "");

    const exists = profiles[url];

    const data = {
      fullName: r.fullName || "",
      firstName: r.firstName || "",
      lastName: r.lastName || "",
      company: r.company || "",
      position: r.position || "",
      location: r.location || "",
      email: r.email || "",
      notes: r.notes || "",
      templateUsed: r.templateUsed || ""
    };

    // --------------------------------------------------
    // CREATE NEW PROFILE
    // --------------------------------------------------
    if (!exists) {
      const messages = buildMessagesFromRow(
        r,
        messageMode,
        chosenTemplate,
        templates
      );

      const templateUsed =
        messageMode === "template"
          ? chosenTemplate || data.templateUsed || ""
          : data.templateUsed || "";

      const tpl = templates[templateUsed] || {};
      const delayDays = Number(tpl.delayDays ?? 3);

profiles[url] = {
  profileUrl: url,
  dateCaptured: nowISO(),
  lastUpdated: nowISO(),

  fullName: data.fullName,
  firstName: data.firstName,
  lastName: data.lastName,
  headline: "",
  company: data.company,
  position: data.position,
  location: data.location,
  connectionStatus: "unknown",

  templateUsed,
  messages,

  sentDates: { msg1: null, msg2: null, msg3: null },

  // ✅ DO NOT enqueue on apply
  nextDueDate: null,

  // ✅ Delay applies only AFTER msg1 is sent
  sequenceDelayDays: delayDays,

  status: "new",
  notes: data.notes,
  email: data.email,
  gtmExportedDate: null
};


      added++;
      continue;
    }

    // --------------------------------------------------
    // EXISTING PROFILE
    // --------------------------------------------------
    if (mode === "skip") {
      skipped++;
      continue;
    }

    // overwrite basic fields (safe merge)
    Object.assign(exists, {
      fullName: data.fullName || exists.fullName,
      firstName: data.firstName || exists.firstName,
      lastName: data.lastName || exists.lastName,
      company: data.company || exists.company,
      position: data.position || exists.position,
      location: data.location || exists.location,
      email: data.email || exists.email,
      notes: data.notes || exists.notes
    });

    const newMessages = buildMessagesFromRow(
      r,
      messageMode,
      chosenTemplate,
      templates
    );

    const hasAnyMessage =
      (newMessages.msg1 || "").trim() ||
      (newMessages.msg2 || "").trim() ||
      (newMessages.msg3 || "").trim();

    if (hasAnyMessage) {
      exists.messages = newMessages;
    } else if (messageMode === "template" && chosenTemplate) {
      exists.messages = JSON.parse(
        JSON.stringify(templates[chosenTemplate]?.messages || {})
      );
      exists.templateUsed = chosenTemplate;
    }

    // --------------------------------------------------
    // RESTART MODE (explicit reset)
    // --------------------------------------------------
    if (mode === "restart") {
      const tpl = templates[exists.templateUsed] || {};
      const delayDays = Number(
        tpl.delayDays ?? exists.sequenceDelayDays ?? 3
      );

      exists.sentDates = { msg1: null, msg2: null, msg3: null };
      exists.status = "new";

      // ✅ Message 1 must re-enter queue
      exists.nextDueDate = null; // wait until msg1 is sent

      // ✅ Keep delay consistent
      exists.sequenceDelayDays = delayDays;
    }

    exists.lastUpdated = nowISO();
    updated++;
  }

  await setLS({ profiles });
  await refreshAll();

  return { added, updated, skipped };
}

  // ===== CSV Template Download =====
  downloadTemplateBtn.onclick = () => {
    const rows = [
      [
        "profileUrl","firstName","lastName","company","position",
        "email","notes","templateUsed","msg1","msg2","msg3"
      ],
      [
        "https://www.linkedin.com/in/example1","John","Doe","Acme",
        "CEO","john@acme.com","Important lead","Template 1",
        "Hi John ...","Following up ...","Final "
      ]
    ];

    const csv = rows.map(r =>
      r.map(c => `"${String(c).replace(/"/g, '""')}"`).join(",")
    ).join("\n");

    const blob = new Blob([csv], { type: "text/csv" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "linkedin_import_template.csv";
    a.click();
    URL.revokeObjectURL(url);
  };

  // ===== Navigation to ICP Setup =====
  document.getElementById("openIcpBtnDashboard")?.addEventListener("click", () => {
    chrome.tabs.create({
      url: chrome.runtime.getURL("icp_setup.html")
    });
  });

  // ===== INITIAL LOAD =====
  await refreshAll();

});
